<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use App\Models\Form;

class FormController extends Controller
{
    /**
     * INDEX PAGE
     */
    public function index()
    {
        // UI listing needs counts for "Total Artworks" and "Mediums" columns.
        $forms = Form::withCount(['paintings', 'mediums'])
            ->orderBy('id', 'desc')
            ->get();

        return view('admin.forms.index', compact('forms'));
    }

    /**
     * STORE (CREATE)
     */
    public function store(Request $request)
    {
        $request->validate([
            'name'   => 'required|string|max:255|unique:forms,name',
            'status' => 'required|in:active,inactive',
        ]);

        Form::create([
            'name'       => $request->name,
            'slug'       => Str::slug($request->name),
            'status'     => $request->status,
            'updated_by' => Auth::guard('admin')->user()->name ?? null,
        ]);

        return redirect()->back()->with('success', 'Form created successfully');
    }

    /**
     * UPDATE (EDIT)
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name'   => 'required|string|max:255|unique:forms,name,' . $id,
            'status' => 'required|in:active,inactive',
        ]);

        $form = Form::findOrFail($id);

        $form->update([
            'name'       => $request->name,
            'slug'       => Str::slug($request->name),
            'status'     => $request->status,
            'updated_by' => Auth::guard('admin')->user()->name ?? null,
        ]);

        return redirect()->back()->with('success', 'Form updated successfully');
    }

    /**
     * DELETE
     */
    public function destroy($id)
    {
        $form = Form::findOrFail($id);

        // Keep admin category flow predictable:
        // a form should not be removed while artworks are attached.
        if ($form->paintings()->count() > 0) {
            return redirect()->back()
                ->with('error', 'Cannot delete form. Paintings are linked to this form.');
        }

        // Prevent orphan mediums that disappear from the form-based Mediums UI.
        if ($form->mediums()->count() > 0) {
            return redirect()->back()
                ->with('error', 'Cannot delete form. Mediums are linked to this form.');
        }

        $form->delete();

        return redirect()->back()->with('success', 'Form deleted successfully');
    }
}
