<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SeoPage;
use App\Models\SeoPageSlugHistory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Validation\ValidationException;

class SeoPageController extends Controller
{
    public function index()
    {
        $seoPages = DB::table('seo_pages')
            ->orderBy('id')
            ->get();

        return view('admin.seo-pages.index', [
            'seoPages' => $seoPages,
            'pageLabels' => $this->pageLabels(),
        ]);
    }

    public function edit(string $key)
    {
        $seoPage = SeoPage::query()
            ->where('page_key', $key)
            ->first();

        abort_if(!$seoPage, 404);

        return view('admin.seo-pages.edit', [
            'seoPage' => $seoPage,
            'pageLabels' => $this->pageLabels(),
        ]);
    }

    public function update(Request $request, string $key)
    {
        $seoPage = SeoPage::query()
            ->where('page_key', $key)
            ->first();

        abort_if(!$seoPage, 404);

        $validated = $request->validate([
            'seo_title' => ['required', 'string', 'max:60'],
            'seo_slug' => [
                'required',
                'string',
                'max:255',
                function ($attribute, $value, $fail) {
                    if (preg_match('/https?:\/\//i', $value) || preg_match('/(^|\/)([a-z0-9-]+\.)+[a-z]{2,}(\/|$)/i', $value)) {
                        $fail('The seo slug must not contain a domain name.');
                    }
                },
            ],
            'meta_description' => ['nullable', 'string', 'max:160'],
            'meta_keywords' => ['nullable', 'array'],
            'meta_keywords.*' => ['nullable', 'string', 'max:50'],
        ]);

        // Store slug without leading slash, e.g. "about-us".
        $newSlug = SeoPage::normalizeSlug((string) $validated['seo_slug']);

        // Backward compatibility: treat "/about-us" and "about-us" as same slug.
        $slugTaken = SeoPage::query()
            ->where('id', '!=', $seoPage->id)
            ->where(function ($q) use ($newSlug) {
                $q->where('seo_slug', $newSlug);
                if ($newSlug !== '') {
                    $q->orWhere('seo_slug', '/' . $newSlug);
                } else {
                    $q->orWhere('seo_slug', '/');
                }
            })
            ->exists();

        if ($slugTaken) {
            throw ValidationException::withMessages([
                'seo_slug' => 'The seo slug has already been taken.',
            ]);
        }

        $keywords = collect($validated['meta_keywords'] ?? [])
            ->map(fn ($item) => trim((string) $item))
            ->filter()
            ->unique()
            ->values()
            ->implode(',');

        DB::transaction(function () use ($seoPage, $validated, $keywords, $newSlug) {
            $oldSlug = SeoPage::normalizeSlug((string) $seoPage->seo_slug);

            // Save history only when effective slug value actually changes.
            if ($oldSlug !== $newSlug && Schema::hasTable('seo_page_slug_histories')) {
                SeoPageSlugHistory::query()->updateOrCreate(
                    ['old_slug' => $oldSlug],
                    ['seo_page_id' => $seoPage->id]
                );
            }

            SeoPage::query()
                ->where('id', $seoPage->id)
                ->update([
                    'seo_title' => $validated['seo_title'],
                    'seo_slug' => $newSlug,
                    'meta_description' => $validated['meta_description'] ?? null,
                    'meta_keywords' => $keywords !== '' ? $keywords : null,
                    'updated_at' => now(),
                ]);
        });

        Cache::forget('seo_page_slug_map');

        return redirect()
            ->route('admin.seo-pages.edit', $seoPage->page_key)
            ->with('success', 'SEO page updated successfully.');
    }

    private function pageLabels(): array
    {
        return [
            'home' => 'Home',
            'about' => 'About',
            'exclusive' => 'Exclusive',
            'artworks' => 'Artworks',
            'artists' => 'Artists',
            'blog' => 'Blog',
            'art_advisory' => 'Art Advisory',
            'contact' => 'Contact',
        ];
    }
}
