<?php

namespace App\Http\Controllers\Front;

use App\Http\Controllers\Controller;
use App\Models\Artist;
use App\Models\Painting;
use Illuminate\Http\Request;
use App\Models\Form;
use App\Models\Category;

class ArtistController extends Controller
{
    /**
     * Artist Listing Page
     */
    public function index(Request $request)
    {
        $search = $request->get('search');
        $letter = $request->get('letter');

        // ⭐ FEATURED ARTISTS
        $featuredArtists = Artist::where('status', 'active')
            ->where('is_featured', 1)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // 👥 OTHER ARTISTS
        $artists = Artist::where('status', 'active')
            ->where(function ($q) {
                $q->whereNull('is_featured')
                  ->orWhere('is_featured', 0);
            })
            ->when($search, function ($query) use ($search) {
                $query->where('name', 'LIKE', '%' . $search . '%');
            })
            ->when($letter, function ($query) use ($letter) {
                $query->where('name', 'LIKE', $letter . '%');
            })
            ->orderBy('name', 'asc')
            ->paginate(12)
            ->withQueryString();

        return view('front.artists.index', compact(
            'featuredArtists',
            'artists',
            'search',
            'letter'
        ));
    }

    /**
     * Artist Detail Page + AJAX Filters
     */
    public function show(Request $request, $slug)
    {
        $artist = Artist::where('slug', $slug)
            ->where('status', 'active')
            ->firstOrFail();

        /* ==========================
         * BASE QUERY (ONLY THIS ARTIST)
         * ========================== */
        $query = Painting::with(['artist', 'categories', 'form'])
            ->where('status', 'active')
            ->where('artist_id', $artist->id);

        /* ==========================
         * PRICE FILTER (FIXED)
         * ========================== */
        if ($request->filled('price_min') || $request->filled('price_max')) {
            $min = $request->filled('price_min') ? (int) $request->price_min : 0;
            $max = $request->filled('price_max') ? (int) $request->price_max : 5000000;

            $query->where(function ($q) use ($min, $max) {
                $q->where('price_on_request', true)
                  ->orWhere(function ($q2) use ($min, $max) {
                      $q2->where('price_on_request', false)
                         ->whereBetween('price', [$min, $max]);
                  });
            });
        }

        /* ==========================
         * FORM FILTER
         * ========================== */
        if ($request->filled('form')) {
            $query->where('form_id', $request->form);
        }

        /* ==========================
         * THEMES FILTER (MULTI)
         * ========================== */
        if ($request->filled('categories')) {
            $query->whereHas('categories', function ($q) use ($request) {
                $q->whereIn('categories.id', $request->categories);
            });
        }

        $paintings = $query->latest()
            ->paginate(12)
            ->withQueryString();

        /* ==========================
         * AJAX REQUEST → ONLY GRID
         * ========================== */
        if ($request->ajax()) {
            return view(
                'front.paintings.partials.painting_grid',
                compact('paintings')
            );
        }

        /* ==========================
         * NORMAL PAGE LOAD
         * ========================== */
        $forms = Form::where('status', 'active')
            ->orderBy('name', 'asc')
            ->get();

        $categories = Category::where('status', 'active')
            ->orderBy('name', 'asc')
            ->get();

        return view('front.artists.show', compact(
            'artist',
            'paintings',
            'forms',
            'categories'
        ));
    }
}
