<?php

namespace App\Http\Controllers\Front;

use App\Http\Controllers\Controller;
use App\Http\Controllers\FrontBlogController;
use App\Models\SeoPage;
use App\Models\SeoPageSlugHistory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class PageController extends Controller
{
    /**
     * Handle all frontend pages via SEO pages
     */
    public function show(Request $request, ?string $slug = null)
    {
        /**
         * If slug is null or empty → treat as home
         */
        $requestedSlug = SeoPage::normalizeSlug($slug ?? '');

        /**
         * If seo_pages table does not exist
         * fallback directly to home
         */
        if (!Schema::hasTable('seo_pages')) {
            return $this->dispatchPage($request, 'home');
        }

        /**
         * Find SEO page
         */
        $seoPage = SeoPage::query()
            ->where(function ($q) use ($requestedSlug) {
                $q->where('seo_slug', $requestedSlug);

                if ($requestedSlug !== '') {
                    $q->orWhere('seo_slug', '/' . $requestedSlug);
                } else {
                    $q->orWhere('seo_slug', '/');
                }
            })
            ->first();

        /**
         * If not found → check slug history (301 redirect)
         */
        if (!$seoPage) {
            if (Schema::hasTable('seo_page_slug_histories')) {
                $history = SeoPageSlugHistory::query()
                    ->where('old_slug', $requestedSlug)
                    ->with('seoPage:id,seo_slug,page_key')
                    ->first();

                if ($history && $history->seoPage) {
                    return redirect(
                        SeoPage::toPath((string) $history->seoPage->seo_slug),
                        301
                    );
                }
            }

            throw new NotFoundHttpException();
        }

        /**
         * Share SEO meta globally
         */
        view()->share('seoMeta', $seoPage);

        /**
         * Always ensure page_key is string
         */
        $pageKey = $seoPage->page_key ?: 'home';

        return $this->dispatchPage($request, $pageKey);
    }

    /**
     * Dispatch page based on page_key
     */
    private function dispatchPage(Request $request, string $pageKey = 'home')
    {
        return match ($pageKey) {
            'home'        => app(FrontController::class)->index(),
            'about'       => app(FrontController::class)->about(),
            'exclusive'   => app(FrontController::class)->exhibitions(),
            'artworks'    => app(PaintingController::class)->index($request),
            'artists'     => app(ArtistController::class)->index($request),
            'blog'        => app(FrontBlogController::class)->index(),
            'art_advisory'=> app(FrontController::class)->artAdvisory(),
            'contact'     => app(FrontController::class)->contact(),
            default       => throw new NotFoundHttpException(),
        };
    }
}
