<?php

namespace App\Http\Controllers\Front;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Painting;
use App\Models\Form;
use App\Models\Category;

class PaintingController extends Controller
{
    /* =====================================================
     * ARTWORK LIST (NORMAL + AJAX + PAGINATION)
     * ===================================================== */
    public function index(Request $request, $slug = null)
    {
        $query = Painting::with(['artist', 'categories', 'form'])
            ->where('status', 'active');

        /* =====================
         * PRICE FILTER (SLIDER)
         * ===================== */
        if ($request->filled('price_min') || $request->filled('price_max')) {
            $min = $request->filled('price_min') ? (int) $request->price_min : 0;
            $max = $request->filled('price_max') ? (int) $request->price_max : 5000000;

            $query->where(function ($q) use ($min, $max) {
                $q->where('price_on_request', true)
                  ->orWhere(function ($q2) use ($min, $max) {
                      $q2->where('price_on_request', false)
                         ->whereBetween('price', [$min, $max]);
                  });
            });
        }


        /* =====================
         * FORM FILTER (DROPDOWN)
         * ===================== */
        if ($request->filled('form')) {
            $query->where('form_id', $request->form);
        }

        /* =====================
         * SEO FORM SLUG
         * /artworks/form/{slug}
         * ===================== */
        if ($slug) {
            $form = Form::where('slug', $slug)->firstOrFail();
            $query->where('form_id', $form->id);
        }

        /* =====================
         * MULTI THEME FILTER
         * ===================== */
        if ($request->filled('categories')) {
            $query->whereHas('categories', function ($q) use ($request) {
                $q->whereIn('categories.id', $request->categories);
            });
        }

        $paintings = $query->latest()
            ->paginate(12)
            ->withQueryString();

        /* =====================
         * AJAX RESPONSE
         * ===================== */
        if ($request->ajax()) {
            return view(
                'front.paintings.partials.painting_grid',
                compact('paintings')
            );
        }

        /* =====================
         * NORMAL PAGE LOAD
         * ===================== */
        $forms = Form::where('status', 'active')
            ->orderBy('name')
            ->get();

        $categories = Category::where('status', 'active')
            ->orderBy('name')
            ->get();

        return view('front.paintings.index', compact(
            'paintings',
            'forms',
            'categories'
        ));
    }

    /* =====================================================
     * ARTWORK DETAIL PAGE
     * ===================================================== */
    public function show($id)
    {
        $painting = Painting::with(['artist', 'categories', 'form'])
            ->where('status', 'active')
            ->findOrFail($id);

        return view('front.paintings.show', compact('painting'));
    }
}
