<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class SeoPage extends Model
{
    protected $fillable = [
        'page_key',
        'seo_title',
        'seo_slug',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'canonical_url',
        'meta_robots',
        'og_title',
        'og_description',
        'og_image',
        'og_type',
        'og_url',
        'twitter_card',
        'twitter_title',
        'twitter_description',
        'twitter_image',
        'schema_markup',
        'breadcrumb_title',
    ];

    public function slugHistories()
    {
        return $this->hasMany(SeoPageSlugHistory::class);
    }

    /**
     * Normalize slug for DB storage.
     * We store values like "about-us" (without leading slash).
     */
    public static function normalizeSlug(string $slug): string
    {
        $slug = trim($slug);
        if ($slug === '' || $slug === '/') {
            return '';
        }

        $slug = ltrim($slug, '/');
        $slug = preg_replace('#/+#', '/', $slug) ?? $slug;

        return trim(Str::lower($slug), '/');
    }

    /**
     * Convert stored slug to public URL path.
     */
    public static function toPath(string $slug): string
    {
        $slug = self::normalizeSlug($slug);
        return $slug === '' ? '/' : '/' . $slug;
    }

    public static function url(string $pageKey, ?string $fallback = null): string
    {
        $map = Cache::remember('seo_page_slug_map', now()->addMinutes(10), function () {
            return self::query()
                ->pluck('seo_slug', 'page_key')
                ->map(fn ($slug) => self::toPath((string) $slug))
                ->all();
        });

        if (!empty($map[$pageKey])) {
            return $map[$pageKey];
        }

        return $fallback ?? '/';
    }
}
