import { ConsentValues, ConsentKeys, StorageVersionV2, StorageVersionV3, DataProcessingPurpose, DisplayConsentValues, ConsentDisplayKeys } from './types/index.mjs';
import { readCookie } from './storage/cookie.mjs';
import { fetchConsentFromURL } from './storage/url.mjs';

function readConsent() {
  const storageObj = fetchConsentFromURL() || readCookie();

  if (storageObj === undefined) {
    return undefined;
  }

  return decodeStorage(storageObj);
} // only exported for testing. Used as a single entrypoint for storage reading,
// it reads tracking consent from multiple supported versions and returns a
// common representation.


function readStorageAndTransform() {
  try {
    let storage = readConsent();

    if (!storage) {
      return undefined;
    }

    return storage;
  } catch {
    return undefined;
  }
}
function getCMPConsentValues() {
  return {
    m: getSignal(ConsentKeys.MARKETING),
    a: getSignal(ConsentKeys.ANALYTICS),
    p: getSignal(ConsentKeys.PREFERENCES),
    s: getSignal(ConsentKeys.SALE_OF_DATA)
  };
}
function getCCPAConsentValue() {
  return getCMPConsentValues()[ConsentKeys.SALE_OF_DATA];
}
function validStorageDoesNotExist(consentStorage = null) {
  if (consentStorage === null) {
    consentStorage = readStorageAndTransform();
  }

  return consentStorage === undefined;
}

function decodeStorage(rawStorage) {
  // The first byte is the version. Before v3 we used JSON, so we use the {
  // char as an informal version identifier.
  const version = rawStorage.slice(0, 1);

  if (version == "{") {
    return decodeV2Storage(rawStorage);
  }

  if (version == "3") {
    return decodeV3Storage(rawStorage);
  }

  return undefined; // unknown version
}

function decodeV2Storage(rawStorage) {
  var _payload$con;

  // Just a JSON Storage
  let payload;

  try {
    payload = JSON.parse(rawStorage);
  } catch {
    return undefined;
  }

  if (payload.v !== StorageVersionV2) {
    return undefined;
  }

  if (!((_payload$con = payload.con) !== null && _payload$con !== void 0 && _payload$con.CMP)) {
    return undefined;
  }

  return payload;
} // See info about the schema at https://core-engage-compliance.docs.shopify.io/components/cmp/browser-cookies


function decodeV3Storage(rawStorage) {
  const fields = rawStorage.slice(1).split("_"); // remove version number and split by field

  const [purposes, region, displayBanner, saleOfDataRegion, encodedConsentId] = fields;
  let cus;

  try {
    cus = fields[5] ? JSON.parse(fields.slice(5).join("_")) : undefined;
  } catch {}

  let consentIdWithHyphens;

  if (encodedConsentId) {
    // We use URL safe base64 encoding (https://ruby-doc.org/stdlib-2.5.3/libdoc/base64/rdoc/Base64.html) but
    // changing "_" by "*".
    // For atob, we need to go back from the URL safe to the "normal" base64 characters.
    const normalBase64 = encodedConsentId.replace(/\*/g, "/").replace(/-/g, "+");
    const consentIdBinary = Array.from(atob(normalBase64));
    const consentIdHex = consentIdBinary.map(char => char.charCodeAt(0).toString(16).padStart(2, "0")).join("");
    consentIdWithHyphens = [8, 13, 18, 23].reduce((result, index) => {
      return result.slice(0, index) + "-" + result.slice(index);
    }, consentIdHex);
  }

  function consentSignal(abbreviation) {
    const purposesWithConsent = purposes.split(".")[0];

    if (purposesWithConsent.includes(abbreviation.toLowerCase())) {
      return ConsentValues.DECLINED;
    } else if (purposesWithConsent.includes(abbreviation.toUpperCase())) {
      return ConsentValues.ACCEPTED;
    } else {
      return ConsentValues.NO_VALUE;
    }
  }

  function purpose(abbreviation) {
    return purposes.includes(abbreviation.replace("t", "s").toUpperCase());
  }

  return {
    v: StorageVersionV3,
    con: {
      CMP: {
        [ConsentKeys.ANALYTICS]: consentSignal(ConsentKeys.ANALYTICS),
        [ConsentKeys.PREFERENCES]: consentSignal(ConsentKeys.PREFERENCES),
        [ConsentKeys.MARKETING]: consentSignal(ConsentKeys.MARKETING),
        [ConsentKeys.SALE_OF_DATA]: consentSignal(ConsentKeys.SALE_OF_DATA)
      }
    },
    region: region || "",
    cus,
    purposes: {
      [DataProcessingPurpose.ANALYTICS]: purpose(DataProcessingPurpose.ANALYTICS),
      [DataProcessingPurpose.PREFERENCES]: purpose(DataProcessingPurpose.PREFERENCES),
      [DataProcessingPurpose.MARKETING]: purpose(DataProcessingPurpose.MARKETING),
      [DataProcessingPurpose.SALE_OF_DATA]: purpose(DataProcessingPurpose.SALE_OF_DATA)
    },
    sale_of_data_region: saleOfDataRegion == "t",
    display_banner: displayBanner == "t",
    consent_id: consentIdWithHyphens
  };
}

function transformConsentToDisplaySchema(consentValue) {
  switch (consentValue) {
    case ConsentValues.ACCEPTED:
      return DisplayConsentValues.ACCEPTED;

    case ConsentValues.DECLINED:
      return DisplayConsentValues.DECLINED;

    default:
      return DisplayConsentValues.NO_VALUE;
  }
}
function transformKeyToDisplayName(key) {
  switch (key) {
    case ConsentKeys.ANALYTICS:
      return ConsentDisplayKeys.ANALYTICS;

    case ConsentKeys.MARKETING:
      return ConsentDisplayKeys.MARKETING;

    case ConsentKeys.PREFERENCES:
      return ConsentDisplayKeys.PREFERENCES;

    case ConsentKeys.SALE_OF_DATA:
      return ConsentDisplayKeys.SALE_OF_DATA;
  }
}
function getSignal(signal) {
  const storage = readStorageAndTransform();

  if (!storage) {
    return ConsentValues.NO_VALUE;
  }

  const signals = storage.con.CMP;

  if (!signals) {
    return ConsentValues.NO_VALUE;
  }

  return signals[signal];
}
function getRegionValue() {
  const storage = readStorageAndTransform();

  if (validStorageDoesNotExist(storage)) {
    return "";
  }

  return storage.region || "";
}
function getCustomValues() {
  const storage = readStorageAndTransform();

  if (validStorageDoesNotExist(storage) || !storage.cus) {
    return undefined;
  }

  return Object.entries(storage.cus).map(([key, value]) => ({
    key: decodeURIComponent(key),
    value: decodeURIComponent(value)
  }));
}
function getCustomValue(key) {
  const storage = readStorageAndTransform();

  if (validStorageDoesNotExist(storage) || !storage.cus) {
    return undefined;
  }

  const val = storage.cus[encodeURIComponent(key)]; // Be careful not to decode undefined, null, etc. They'll get promoted to strings otherwise.

  return val ? decodeURIComponent(val) : val;
}

function getPurposeFromStorage(purpose) {
  const storage = readConsent();

  if (!storage || !storage.purposes) {
    return true;
  }

  const purposeValue = storage.purposes[purpose];
  return typeof purposeValue === "boolean" ? purposeValue : true;
}

function preferencesProcessingAllowed() {
  return getPurposeFromStorage(DataProcessingPurpose.PREFERENCES);
}
function analyticsProcessingAllowed() {
  return getPurposeFromStorage(DataProcessingPurpose.ANALYTICS);
}
function marketingProcessingAllowed() {
  return getPurposeFromStorage(DataProcessingPurpose.MARKETING);
}
function saleOfDataAllowed() {
  return getPurposeFromStorage(DataProcessingPurpose.SALE_OF_DATA);
}
function shouldShowBanner() {
  const storage = readConsent();

  if (!storage) {
    return false;
  }

  return typeof storage.display_banner === "boolean" ? storage.display_banner : false;
}
function saleOfDataRegion() {
  const storage = readConsent();

  if (!storage) {
    return false;
  }

  return storage.sale_of_data_region || false;
}
function getConsentId() {
  const storage = readConsent();

  if (!storage) {
    return "";
  }

  return storage.consent_id || "";
}

export { analyticsProcessingAllowed, getCCPAConsentValue, getCMPConsentValues, getConsentId, getCustomValue, getCustomValues, getRegionValue, getSignal, marketingProcessingAllowed, preferencesProcessingAllowed, readStorageAndTransform, saleOfDataAllowed, saleOfDataRegion, shouldShowBanner, transformConsentToDisplaySchema, transformKeyToDisplayName, validStorageDoesNotExist };
