import { objectSpread2 as _objectSpread2 } from '../_virtual/_rollupPluginBabelHelpers.mjs';
import { extractDomain, DEVELOPMENT_DOMAIN, PRODUCE_BATCH_ENDPOINT, PRODUCE_ENDPOINT, PRODUCTION_DOMAIN } from '../config/endpoints.mjs';
import { toUnderscoreCase, convertAllEventsToUnderscoreCase } from '../helpers/convert-case.mjs';
import { v4 } from '../helpers/uuid.mjs';
import { MonorailRequestError, MonorailBatchProduceError, MonorailResponseReadError, MonorailInterceptedProduceError, MonorailUnableToProduceError } from './producer-errors.mjs';

class HttpProducer {
  static withEndpoint(edgeEndpoint) {
    return new HttpProducer(extractDomain(edgeEndpoint));
  }
  constructor(edgeDomain = DEVELOPMENT_DOMAIN, optionsOrKeepalive = {}) {
    var _optionsOrKeepalive$k, _optionsOrKeepalive$d;
    this.edgeDomain = edgeDomain;
    this.optionsOrKeepalive = optionsOrKeepalive;
    if (typeof optionsOrKeepalive === 'boolean') {
      this.keepalive = optionsOrKeepalive;
      this.detectInterceptedErrorEnabled = false;
      return;
    }
    this.keepalive = (_optionsOrKeepalive$k = optionsOrKeepalive.keepalive) !== null && _optionsOrKeepalive$k !== void 0 ? _optionsOrKeepalive$k : false;
    this.detectInterceptedErrorEnabled = (_optionsOrKeepalive$d = optionsOrKeepalive.detectInterceptedErrorEnabled) !== null && _optionsOrKeepalive$d !== void 0 ? _optionsOrKeepalive$d : false;
  }
  async produceBatch(monorailEvent) {
    const body = {
      events: convertAllEventsToUnderscoreCase(monorailEvent),
      metadata: toUnderscoreCase(monorailEvent.metadata)
    };
    let edgeResponse;
    try {
      edgeResponse = await fetch(this.produceBatchEndpoint(), {
        method: 'post',
        headers: getHeadersFromMetadata(monorailEvent.metadata),
        body: JSON.stringify(body),
        keepalive: this.keepalive
      });
    } catch (error) {
      throw new MonorailRequestError(error, monorailEvent.events.map(event => event.schemaId));
    }
    if (edgeResponse.status === 207) {
      const response = await edgeResponse.json();
      throw new MonorailBatchProduceError(response);
    }
    let responseText;
    try {
      responseText = await edgeResponse.text();
    } catch (error) {
      throw new MonorailResponseReadError(error, edgeResponse.status);
    }
    if (!edgeResponse.ok) {
      const isFromEdgeServer = Boolean(edgeResponse.headers.get('x-request-id'));
      if (!isFromEdgeServer && this.detectInterceptedErrorEnabled) {
        throw new MonorailInterceptedProduceError({
          status: edgeResponse.status,
          message: responseText
        });
      }
      throw new MonorailUnableToProduceError({
        status: edgeResponse.status,
        message: responseText
      });
    }
    return {
      status: edgeResponse.status
    };
  }
  async produce(monorailEvent) {
    let convertToUnderscoreCase = true;
    if (monorailEvent && monorailEvent.options && Object.prototype.hasOwnProperty.call(monorailEvent.options, 'convertEventCase')) {
      convertToUnderscoreCase = Boolean(monorailEvent.options.convertEventCase);
    }
    let edgeResponse;
    try {
      edgeResponse = await produceHttpEvent({
        endpoint: this.produceEndpoint(),
        keepalive: this.keepalive,
        event: _objectSpread2(_objectSpread2({}, monorailEvent), {}, {
          payload: toUnderscoreCase(monorailEvent.payload, convertToUnderscoreCase)
        })
      });
    } catch (error) {
      throw new MonorailRequestError(error, [monorailEvent.schemaId]);
    }
    if (!edgeResponse) {
      throw new MonorailUnableToProduceError({
        message: 'No response from edge'
      });
    }
    let responseText;
    try {
      responseText = await edgeResponse.text();
    } catch (error) {
      throw new MonorailResponseReadError(error, edgeResponse.status);
    }
    if (!edgeResponse.ok) {
      const isFromEdgeServer = Boolean(edgeResponse.headers.get('x-request-id'));
      if (!isFromEdgeServer && this.detectInterceptedErrorEnabled) {
        throw new MonorailInterceptedProduceError({
          status: edgeResponse.status,
          message: responseText
        });
      }
      throw new MonorailUnableToProduceError({
        status: edgeResponse.status,
        message: responseText
      });
    }
    return {
      status: edgeResponse.status
    };
  }
  produceBatchEndpoint() {
    return this.edgeDomain + PRODUCE_BATCH_ENDPOINT;
  }
  produceEndpoint() {
    return this.edgeDomain + PRODUCE_ENDPOINT;
  }
}
function getHeadersFromMetadata(metadata) {
  /* eslint-disable @typescript-eslint/naming-convention */
  const headers = {
    'Content-Type': 'application/json; charset=utf-8',
    'X-Monorail-Edge-Event-Created-At-Ms': (metadata && metadata.eventCreatedAtMs || Date.now()).toString(),
    'X-Monorail-Edge-Event-Sent-At-Ms': Date.now().toString(),
    'X-Monorail-Edge-Client-Message-Id': (metadata && metadata.clientMessageId || v4()).toString()
  };
  /* eslint-enable @typescript-eslint/naming-convention */

  if (metadata && metadata.userAgent) {
    headers['User-Agent'] = metadata.userAgent;
  }
  if (metadata && metadata.remoteIp) {
    headers['X-Forwarded-For'] = metadata.remoteIp;
  }
  if (metadata && metadata.deviceInstallId) {
    headers['X-Monorail-Edge-Device-Install-Id'] = metadata.deviceInstallId;
  }
  if (metadata && metadata.client) {
    headers['X-Monorail-Edge-Client'] = metadata.client;
  }
  if (metadata && metadata.clientOs) {
    headers['X-Monorail-Edge-Client-OS'] = metadata.clientOs;
  }
  return headers;
}

// Allows avoiding HttpProducer entirely by passing in snake-cased events (but we still get typechecking)
async function produceHttpEvent({
  endpoint,
  event,
  keepalive
}) {
  var _event$metadata, _event$metadata2, _event$metadata3, _event$metadata4, _event$metadata5;
  const filteredMetadata = event.metadata ? {
    clientMessageId: (_event$metadata = event.metadata) === null || _event$metadata === void 0 ? void 0 : _event$metadata.clientMessageId,
    eventCreatedAtMs: (_event$metadata2 = event.metadata) === null || _event$metadata2 === void 0 ? void 0 : _event$metadata2.eventCreatedAtMs,
    consent: (_event$metadata3 = event.metadata) === null || _event$metadata3 === void 0 ? void 0 : _event$metadata3.consent,
    /* eslint-disable @typescript-eslint/naming-convention */
    consent_provider: (_event$metadata4 = event.metadata) === null || _event$metadata4 === void 0 ? void 0 : _event$metadata4.consent_provider,
    consent_version: (_event$metadata5 = event.metadata) === null || _event$metadata5 === void 0 ? void 0 : _event$metadata5.consent_version
    /* eslint-enable @typescript-eslint/naming-convention */
  } : undefined;
  return fetch(endpoint !== null && endpoint !== void 0 ? endpoint : PRODUCTION_DOMAIN + PRODUCE_ENDPOINT, {
    method: 'post',
    headers: getHeadersFromMetadata(event.metadata),
    body: JSON.stringify({
      /* eslint-disable-next-line @typescript-eslint/naming-convention */
      schema_id: event.schemaId,
      payload: event.payload,
      metadata: filteredMetadata && toUnderscoreCase(filteredMetadata, true)
    }),
    keepalive
  });
}

export { HttpProducer, produceHttpEvent };
