import { defineProperty as _defineProperty } from '../_virtual/_rollupPluginBabelHelpers.mjs';

/*
Please note that in both of the super calls below we are setting
the prototype chain manually. This needs to be done when extending built-ins like
`Error`, `Array` and `Map`.
The detailed reason why we are doing this is because of the following recommendations
from the typescript team:
https://github.com/Microsoft/TypeScript/wiki/FAQ#why-doesnt-extending-built-ins-like-error-array-and-map-work
 */

class MonorailUnableToProduceError extends Error {
  constructor(response) {
    super(`Error producing to the Monorail Edge. Response received: ${JSON.stringify(response)}`);
    _defineProperty(this, "name", 'MonorailUnableToProduceError');
    this.response = response;
    Object.setPrototypeOf(this, MonorailUnableToProduceError.prototype);
  }
}
class MonorailInterceptedProduceError extends Error {
  constructor(response) {
    super(`Response not from Monorail Edge. Response received: ${JSON.stringify(response)}`);
    _defineProperty(this, "name", 'MonorailInterceptedProduceError');
    this.response = response;
    Object.setPrototypeOf(this, MonorailInterceptedProduceError.prototype);
  }
}
class MonorailBatchProduceError extends Error {
  constructor(response) {
    super(`Error producing to the Monorail Edge. Response received: ${JSON.stringify(response)}`);
    _defineProperty(this, "name", 'MonorailBatchProduceError');
    Object.setPrototypeOf(this, MonorailBatchProduceError.prototype);
    this.response = response;
  }
}
class MonorailRequestError extends Error {
  constructor(error, schemaIds) {
    super(`Error completing request. A network failure may have prevented the request from completing. Error: ${error}. Schemas: ${Array.from(new Set(schemaIds)).join(', ')}`);
    _defineProperty(this, "name", 'MonorailRequestError');
    Object.setPrototypeOf(this, MonorailRequestError.prototype);
  }
}
class MonorailResponseReadError extends Error {
  constructor(error, status) {
    super(`Error reading response from Monorail Edge. Status: ${status || 'unknown'}. Error: ${(error === null || error === void 0 ? void 0 : error.message) || 'Unknown error'}`);
    _defineProperty(this, "name", 'MonorailResponseReadError');
    this.error = error;
    this.status = status;
    Object.setPrototypeOf(this, MonorailResponseReadError.prototype);
  }
}
class MonorailRetriesExceededError extends Error {
  constructor(error) {
    super(`${error}`);
    _defineProperty(this, "name", 'MonorailRetriesExceededError');
    Object.setPrototypeOf(this, MonorailRetriesExceededError.prototype);
  }
}

// Parse an error message to find schema_id
// See: lang/go/server/edge/errors.go
function parseServerSchemaError(errorMessage) {
  var _errorMessage$match;
  return errorMessage === null || errorMessage === void 0 || (_errorMessage$match = errorMessage.match(/\[schema_id=([^\]]+)\]/)) === null || _errorMessage$match === void 0 ? void 0 : _errorMessage$match.at(1);
}

export { MonorailBatchProduceError, MonorailInterceptedProduceError, MonorailRequestError, MonorailResponseReadError, MonorailRetriesExceededError, MonorailUnableToProduceError, parseServerSchemaError };
