import { classPrivateFieldLooseBase as _classPrivateFieldLooseBase, classPrivateFieldLooseKey as _classPrivateFieldLooseKey } from './_virtual/_rollupPluginBabelHelpers.mjs';
import { asTypedAttributes, asStringAttributes } from './util.mjs';

const AGGREGATION_TEMPORALITY_DELTA = 1;
const HISTOGRAM_DEFAULT_BOUNDS = buildExplicitBounds(5, 2, 12);
function buildExplicitBounds(seed, base, max) {
  const values = [0];
  for (let x = 0; x < max; x++) {
    const value = Math.floor(seed * base ** x);
    values.push(value);
  }
  return values;
}

/**
 * Attributes passed to metrics should be low-cardinality.
 * Only primitive values are supported.
 */
var _exporter = /*#__PURE__*/_classPrivateFieldLooseKey("exporter");
var _attributes = /*#__PURE__*/_classPrivateFieldLooseKey("attributes");
var _metrics = /*#__PURE__*/_classPrivateFieldLooseKey("metrics");
var _logs = /*#__PURE__*/_classPrivateFieldLooseKey("logs");
/*
 * OpenTelemetryClient is a minimal client for aggregating and sending metrics to an exporter.
 *
 * This module should have no dependencies other than `fetch` and `JSON`.
 *
 * The client should be constructed with an exporter. The client's record method is used to record discrete events from
 * the application. Once events are recorded, the client's exportMetrics method can be called to export metrics to the
 * exporter. The recorded events are aggregated by their name as data points.
 *
 */
class OpenTelemetryClient {
  /*
   * @param exporter - An implementation of an OpenTelemetryMetricsExporter.
   * @param resourceAttributes - Optional. Attributes that will apply to all exported metrics.
   */
  constructor({
    exporter,
    attributes
  }) {
    Object.defineProperty(this, _exporter, {
      writable: true,
      value: void 0
    });
    Object.defineProperty(this, _attributes, {
      writable: true,
      value: void 0
    });
    Object.defineProperty(this, _metrics, {
      writable: true,
      value: []
    });
    Object.defineProperty(this, _logs, {
      writable: true,
      value: []
    });
    _classPrivateFieldLooseBase(this, _exporter)[_exporter] = exporter;
    _classPrivateFieldLooseBase(this, _attributes)[_attributes] = attributes !== null && attributes !== void 0 ? attributes : {};
  }

  /*
   * addAttributes adds attributes to that will be applied to all exported metrics.
   *
   * @param attributes - Attributes to apply to all exported metrics.
   */
  addAttributes(attributes) {
    _classPrivateFieldLooseBase(this, _attributes)[_attributes] = {
      ..._classPrivateFieldLooseBase(this, _attributes)[_attributes],
      ...attributes
    };
  }

  /*
   * histogram records a data point for a Histogram metric.
   *
   * @param name - The name of the histogram being recorded.
   * @param value - The value of the measurement.
   * @param unit - Optional. The unit of measurement, defaults to '1'.
   * @param bounds - Optional. Explicit bounds to use for the histogram buckets.
   *   When provided, the histogram will be encoded as a "conventional" histogram
   *   with the specified buckets. When not provided, the histogram will be encoded
   *   as an "exponential" / "native" histogram, which is typically much more accurate.
   *   To replicate previous library behavior of "conventional histogram with default buckets",
   *   provide the HISTOGRAM_DEFAULT_BOUNDS constant as the buckets parameter.
   * @param scale - Optional. For an exponential histogram, modifies
   *   the base "scale" factor used to calculate implicit bucket
   *   boundaries. The default value - 3 - represents a good
   *   tradeoff between bucket sizes and space usage, and should
   *   suffice for most cases. This parameter is ignored if the
   *   "buckets" parameter is also passed, because it has no effect
   *   for a "conventional" / "bucketed" histogram.
   * @param attributes - Optional. Attributes to apply to the data point.
   */
  histogram({
    name,
    value,
    unit,
    bounds,
    attributes,
    scale
  }) {
    const timeUnixNano = Date.now() * 1000000;
    if (bounds) {
      _classPrivateFieldLooseBase(this, _metrics)[_metrics].push({
        name,
        type: 'histogram',
        value,
        unit,
        timeUnixNano,
        attributes,
        bounds
      });
    } else {
      _classPrivateFieldLooseBase(this, _metrics)[_metrics].push({
        name,
        type: 'exponential_histogram',
        value,
        unit,
        timeUnixNano,
        attributes,
        scale
      });
    }
  }

  /*
   * counter records a data point for a Counter metric.
   *
   * @param name - The name of the counter being recorded.
   * @param value - The value of the measurement.
   * @param unit - Optional. The unit of measurement, defaults to '1'.
   * @param attributes - Optional. Attributes to apply to the data point.
   */
  counter({
    name,
    value,
    unit,
    attributes
  }) {
    const timeUnixNano = Date.now() * 1000000;
    _classPrivateFieldLooseBase(this, _metrics)[_metrics].push({
      name,
      type: 'counter',
      value,
      unit,
      timeUnixNano,
      attributes
    });
  }

  /*
   * gauge records a data point for a Gauge metric.
   *
   * @param name - The name of the gauge being recorded.
   * @param value - The value of the measurement.
   * @param unit - Optional. The unit of measurement, defaults to '1'.
   * @param attributes - Optional. Attributes to apply to the data point.
   */
  gauge({
    name,
    value,
    unit,
    attributes
  }) {
    const timeUnixNano = Date.now() * 1000000;
    _classPrivateFieldLooseBase(this, _metrics)[_metrics].push({
      name,
      type: 'gauge',
      value,
      unit,
      timeUnixNano,
      attributes
    });
  }

  /*
   *
   * log records a log message.
   *
   * @param body - Optional. The actual log message.
   * @param attributes - Optional. Attributes to apply to the log message.
   */
  log({
    body,
    attributes
  }) {
    const timeUnixNano = Date.now() * 1000000;
    _classPrivateFieldLooseBase(this, _logs)[_logs].push({
      timeUnixNano,
      body,
      attributes
    });
  }

  /*
   * exportMetrics builds the aggregated metrics and exports them.
   */
  async exportMetrics() {
    const metricsByName = {};
    _classPrivateFieldLooseBase(this, _metrics)[_metrics].forEach(metric => {
      metric.attributes = {
        ..._classPrivateFieldLooseBase(this, _attributes)[_attributes],
        ...metric.attributes
      };
      switch (metric.type) {
        case 'histogram':
          addHistogram(metricsByName, metric);
          break;
        case 'exponential_histogram':
          addExponentialHistogram(metricsByName, metric);
          break;
        case 'counter':
          addCounter(metricsByName, metric);
          break;
        case 'gauge':
          addGauge(metricsByName, metric);
          break;
      }
    });
    const metrics = Object.values(metricsByName);
    if (metrics.length === 0) {
      return;
    }

    // Clear metrics so that later calls to exportMetrics won't export the same metrics
    _classPrivateFieldLooseBase(this, _metrics)[_metrics] = [];
    await _classPrivateFieldLooseBase(this, _exporter)[_exporter].exportMetrics(metrics);
  }

  /*
   * exportLogs exports the aggregated log records.
   */
  async exportLogs() {
    const logs = _classPrivateFieldLooseBase(this, _logs)[_logs].map(log => {
      const exportLog = {
        timeUnixNano: log.timeUnixNano,
        observedTimeUnixNano: log.timeUnixNano,
        attributes: asTypedAttributes({
          ..._classPrivateFieldLooseBase(this, _attributes)[_attributes],
          ...log.attributes
        })
      };
      if (log.body) {
        exportLog.body = {
          stringValue: log.body
        };
      }
      return exportLog;
    });
    if (logs.length === 0) {
      return;
    }
    _classPrivateFieldLooseBase(this, _logs)[_logs] = [];
    await _classPrivateFieldLooseBase(this, _exporter)[_exporter].exportLogs(logs);
  }
}

// Histogram format: https://github.com/open-telemetry/opentelemetry-proto/blob/ac3242b03157295e4ee9e616af53b81517b06559/opentelemetry/proto/metrics/v1/metrics.proto#L219-L227
function addHistogram(metricsByName, metric) {
  var _metric$bounds;
  const {
    name,
    value,
    unit,
    timeUnixNano,
    attributes
  } = metric;
  const explicitBounds = (_metric$bounds = metric.bounds) !== null && _metric$bounds !== void 0 ? _metric$bounds : HISTOGRAM_DEFAULT_BOUNDS;

  // The number of buckets is one more than the number of explicit bounds.
  const bucketCounts = new Array(explicitBounds.length + 1).fill(0);

  // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
  metricsByName[name] ||= {
    name,
    unit: unit || '1',
    histogram: {
      aggregationTemporality: AGGREGATION_TEMPORALITY_DELTA,
      dataPoints: []
    }
  };
  for (let index = 0; index < bucketCounts.length; index++) {
    const boundary = explicitBounds[index];

    // Increment the bucket count if the value is less than or equal to the boundary.
    // The last bucket's boundary is +Infinity, but is not included in the bounds array,
    // so if the boundary is undefined, then the value must be in the last bucket.
    if (boundary === undefined) {
      bucketCounts[index] = 1;
    } else if (value <= boundary) {
      bucketCounts[index] = 1;
      break;
    }
  }
  metricsByName[name].histogram.dataPoints.push({
    startTimeUnixNano: timeUnixNano,
    timeUnixNano,
    count: 1,
    sum: value,
    min: value,
    max: value,
    bucketCounts,
    explicitBounds,
    attributes: asStringAttributes(attributes !== null && attributes !== void 0 ? attributes : {})
  });
}

// Exponential Histogram format: https://github.com/open-telemetry/opentelemetry-proto/blob/ac3242b03157295e4ee9e616af53b81517b06559/opentelemetry/proto/metrics/v1/metrics.proto#L229-L237
function addExponentialHistogram(metricsByName, metric) {
  const {
    name,
    value,
    unit,
    timeUnixNano,
    attributes,
    scale
  } = metric;

  // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
  metricsByName[name] ||= {
    name,
    unit: unit || '1',
    exponentialHistogram: {
      aggregationTemporality: AGGREGATION_TEMPORALITY_DELTA,
      dataPoints: []
    }
  };
  const valueOrZero = value <= 0 ? 0 : value;
  const scaleOrDefault = scale || 3;
  const scaleFactor = 2 ** scaleOrDefault / Math.log(2);
  const index = Math.ceil(Math.log(value) * scaleFactor) - 1;

  // xTODO: properly support negative observations. It's not clear if this is actually widely supported or not.
  const zeroCount = value <= 0 ? 1 : 0;
  const negativeBuckets = {
    offset: 0,
    bucketCounts: []
  };
  const positiveBuckets = {
    offset: value > 0 ? index : 0,
    bucketCounts: value > 0 ? [1] : []
  };
  metricsByName[name].exponentialHistogram.dataPoints.push({
    attributes: asStringAttributes(attributes !== null && attributes !== void 0 ? attributes : {}),
    startTimeUnixNano: timeUnixNano,
    timeUnixNano,
    count: 1,
    sum: valueOrZero,
    scale: scaleOrDefault,
    zeroCount,
    positive: positiveBuckets,
    negative: negativeBuckets,
    min: valueOrZero,
    max: valueOrZero,
    zeroThreshold: 0
  });
}

// Counter (Sum) format: https://github.com/open-telemetry/opentelemetry-proto/blob/ac3242b03157295e4ee9e616af53b81517b06559/opentelemetry/proto/metrics/v1/metrics.proto#L206-L217
function addCounter(metricsByName, metric) {
  const {
    name,
    value,
    unit,
    timeUnixNano,
    attributes
  } = metric;

  // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
  metricsByName[name] ||= {
    name,
    unit: unit || '1',
    sum: {
      aggregationTemporality: AGGREGATION_TEMPORALITY_DELTA,
      isMonotonic: true,
      dataPoints: []
    }
  };
  metricsByName[name].sum.dataPoints.push({
    startTimeUnixNano: timeUnixNano,
    timeUnixNano,
    asDouble: value,
    attributes: asStringAttributes(attributes !== null && attributes !== void 0 ? attributes : {})
  });
}

// Gauge format: https://github.com/open-telemetry/opentelemetry-proto/blob/ac3242b03157295e4ee9e616af53b81517b06559/opentelemetry/proto/metrics/v1/metrics.proto#L193-L204
function addGauge(metricsByName, metric) {
  const {
    name,
    value,
    unit,
    timeUnixNano,
    attributes
  } = metric;

  // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
  metricsByName[name] ||= {
    name,
    unit: unit || '1',
    gauge: {
      dataPoints: []
    }
  };
  metricsByName[name].gauge.dataPoints.push({
    startTimeUnixNano: timeUnixNano,
    timeUnixNano,
    asDouble: value,
    attributes: asStringAttributes(attributes !== null && attributes !== void 0 ? attributes : {})
  });
}

export { HISTOGRAM_DEFAULT_BOUNDS, OpenTelemetryClient, buildExplicitBounds };
