import { classPrivateFieldLooseBase as _classPrivateFieldLooseBase, classPrivateFieldLooseKey as _classPrivateFieldLooseKey } from './_virtual/_rollupPluginBabelHelpers.mjs';

var _url = /*#__PURE__*/_classPrivateFieldLooseKey("url");
var _serviceName = /*#__PURE__*/_classPrivateFieldLooseKey("serviceName");
var _logger = /*#__PURE__*/_classPrivateFieldLooseKey("logger");
var _fetchFn = /*#__PURE__*/_classPrivateFieldLooseKey("fetchFn");
/*
 * OpenTelemetryMetricsExporter is an interface for exporting metrics. An exporter must be provided
 * to instances of OpenTelemetryClient. The client will call the exporter's exportMetrics method
 * when the client's exportMetrics method is called.
 */

/*
 * OpenTelemetryLogsExporter is an interface for exporting logs. An exporter must be provided
 * to instances of OpenTelemetryClient. The client will call the exporter's exportLogs method
 * when the client's exportLogs method is called.
 */

/*
 * OpenTelemetryJSONExporter is an exporter for sending metrics to an OpenTelemetry Collector.
 *
 * See also: [OpenTelemetry JSON API](https://github.com/open-telemetry/oteps/blob/main/text/0122-otlp-http-json.md)
 */
class OpenTelemetryJSONExporter {
  /**
   * Creates a new instance of the OTLP JSON exporter.
   *
   * @param url - The OpenTelementry collector endpoint where data will be sent.
   * @param serviceName - The name of the service that is using this exporter.
   * @param logger - Optional. A logger that will be called with event objects.
   * @param fetchFn - Optional. Custom fetch function to use when sending a request.
   */
  constructor(url, serviceName, options) {
    Object.defineProperty(this, _url, {
      writable: true,
      value: void 0
    });
    Object.defineProperty(this, _serviceName, {
      writable: true,
      value: void 0
    });
    Object.defineProperty(this, _logger, {
      writable: true,
      value: void 0
    });
    Object.defineProperty(this, _fetchFn, {
      writable: true,
      value: void 0
    });
    _classPrivateFieldLooseBase(this, _url)[_url] = url.replace(/\/v1\/(logs|metrics|traces)\/?$/, '');
    _classPrivateFieldLooseBase(this, _serviceName)[_serviceName] = serviceName;
    _classPrivateFieldLooseBase(this, _logger)[_logger] = options === null || options === void 0 ? void 0 : options.logger;
    _classPrivateFieldLooseBase(this, _fetchFn)[_fetchFn] = options === null || options === void 0 ? void 0 : options.fetchFn;
  }

  /*
   * Export metrics to a JSON collector.
   *
   * The format for the request body is documented here:
   * https://github.com/open-telemetry/opentelemetry-proto/blob/ac3242b03157295e4ee9e616af53b81517b06559/opentelemetry/proto/metrics/v1/metrics.proto
   *
   * @param metrics - A collection of metric objects to export.
   */
  async exportMetrics(metrics) {
    const body = {
      resourceMetrics: [{
        resource: {
          attributes: [{
            key: 'service.name',
            value: {
              stringValue: _classPrivateFieldLooseBase(this, _serviceName)[_serviceName]
            }
          }]
        },
        scopeMetrics: [{
          scope: {
            name: 'open-telemetry-mini-client',
            version: '1.1.0',
            attributes: []
          },
          metrics
        }]
      }]
    };
    await this.exportTo(body, '/v1/metrics');
  }

  /*
   * Export logs to a JSON collector.
   *
   * The format for the request body is documented here:
   * https://github.com/open-telemetry/opentelemetry-proto/blob/ac3242b03157295e4ee9e616af53b81517b06559/opentelemetry/proto/metrics/v1/metrics.proto
   *
   * @param metrics - A collection of metric objects to export.
   */
  async exportLogs(logs) {
    const body = {
      resourceLogs: [{
        resource: {
          attributes: [{
            key: 'service.name',
            value: {
              stringValue: _classPrivateFieldLooseBase(this, _serviceName)[_serviceName]
            }
          }]
        },
        scopeLogs: [{
          scope: {
            name: 'open-telemetry-mini-client',
            version: '1.1.0',
            attributes: []
          },
          logRecords: logs
        }]
      }]
    };
    await this.exportTo(body, '/v1/logs');
  }
  async exportTo(body, suffix) {
    var _classPrivateFieldLoo;
    const response = await this.exporterFetch()(`${_classPrivateFieldLooseBase(this, _url)[_url]}${suffix}`, {
      method: 'POST',
      keepalive: true,
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(body)
    });
    (_classPrivateFieldLoo = _classPrivateFieldLooseBase(this, _logger)[_logger]) === null || _classPrivateFieldLoo === void 0 ? void 0 : _classPrivateFieldLoo.log({
      status: response.status
    });

    // On success the server MUST respond with HTTP 200 OK. Response body MUST be JSON-encoded ExportTraceServiceResponse
    // message for traces and ExportMetricsServiceResponse message for metrics.
    if (response.ok) {
      return;
    }

    // If the processing of the request fails because the request contains data that cannot be decoded or is otherwise
    // invalid and such failure is permanent then the server MUST respond with HTTP 400 Bad Request.
    // The Status.details field in the response SHOULD contain a BadRequest that describes the bad data.
    // The client MUST NOT retry the request when it receives HTTP 400 Bad Request response.
    if (response.status === 400) {
      const errorDetail = await response.text();
      throw new OpenTelemetryClientError(`Invalid OpenTelemetry Data: ${errorDetail}`);
    }

    // If the server receives more requests than the client is allowed or the server is overloaded the server SHOULD
    // respond with HTTP 429 Too Many Requests or HTTP 503 Service Unavailable and MAY include "Retry-After" header
    // with a recommended time interval in seconds to wait before retrying.
    if (response.status === 429 || response.status === 503) {
      const errorData = await response.text();
      const retryHeader = response.headers.get('Retry-After');
      const retryAfter = retryHeader ? {
        seconds: Number(retryHeader)
      } : undefined;
      throw new OpenTelemetryClientError(`Server did not accept data`, {
        errorData,
        retryAfter,
        body
      });
    }
    throw new OpenTelemetryClientError(`Server responded with ${response.status}`);
  }
  exporterFetch() {
    return _classPrivateFieldLooseBase(this, _fetchFn)[_fetchFn] || fetch;
  }
}
class OpenTelemetryClientError extends Error {
  constructor(message, metadata) {
    super(message);
    this.metadata = void 0;
    this.name = 'OpenTelemetryClientError';
    this.metadata = metadata;
  }
}

export { OpenTelemetryClientError, OpenTelemetryJSONExporter };
